<?php
/**
 * Copyright (C) 2014-2020 ServMask Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * ███████╗███████╗██████╗ ██╗   ██╗███╗   ███╗ █████╗ ███████╗██╗  ██╗
 * ██╔════╝██╔════╝██╔══██╗██║   ██║████╗ ████║██╔══██╗██╔════╝██║ ██╔╝
 * ███████╗█████╗  ██████╔╝██║   ██║██╔████╔██║███████║███████╗█████╔╝
 * ╚════██║██╔══╝  ██╔══██╗╚██╗ ██╔╝██║╚██╔╝██║██╔══██║╚════██║██╔═██╗
 * ███████║███████╗██║  ██║ ╚████╔╝ ██║ ╚═╝ ██║██║  ██║███████║██║  ██╗
 * ╚══════╝╚══════╝╚═╝  ╚═╝  ╚═══╝  ╚═╝     ╚═╝╚═╝  ╚═╝╚══════╝╚═╝  ╚═╝
 */

if ( ! defined( 'ABSPATH' ) ) {
	die( 'Kangaroos cannot jump here' );
}

class Ai1wmae_Export_Upload {

	public static function execute( $params, Ai1wmae_B2_Client $b2 = null ) {

		$params['completed'] = false;

		// Set archive offset
		if ( ! isset( $params['archive_offset'] ) ) {
			$params['archive_offset'] = 0;
		}

		// Set multipart offset
		if ( ! isset( $params['multipart_offset'] ) ) {
			$params['multipart_offset'] = 0;
		}

		// Set archive size
		if ( ! isset( $params['archive_size'] ) ) {
			$params['archive_size'] = ai1wm_archive_bytes( $params );
		}

		// Set file chunk number
		if ( ! isset( $params['file_chunk_number'] ) ) {
			$params['file_chunk_number'] = 1;
		}

		// Set start retries
		if ( ! isset( $params['start_retries'] ) ) {
			$params['start_retries'] = 0;
		}

		// Set upload retries
		if ( ! isset( $params['upload_retries'] ) ) {
			$params['upload_retries'] = 0;
		}

		// Set upload backoff
		if ( ! isset( $params['upload_backoff'] ) ) {
			$params['upload_backoff'] = 1;
		}

		// Set Backblaze B2 client
		if ( is_null( $b2 ) ) {
			$b2 = new Ai1wmae_B2_Client(
				get_option( 'ai1wmae_b2_account_id', false ),
				get_option( 'ai1wmae_b2_application_key', false )
			);
		}

		$b2->authorize_account();

		// Open the archive file for reading
		$archive = fopen( ai1wm_archive_path( $params ), 'rb' );

		// Set file chunk size for upload
		$file_chunk_size = get_option( 'ai1wmae_b2_file_chunk_size', AI1WMAE_DEFAULT_FILE_CHUNK_SIZE );

		// Read file chunk data
		if ( ( fseek( $archive, $params['archive_offset'] ) !== -1 )
				&& ( $file_chunk_data = fread( $archive, $file_chunk_size ) ) ) {

			if ( $params['archive_size'] <= $file_chunk_size ) {
				try {

					$params['upload_retries'] += 1;

					// Get new upload part URL
					if ( ( $upload_part = $b2->get_upload_url( $params['bucket_id'] ) ) ) {
						if ( isset( $upload_part['uploadUrl'] ) ) {
							$b2->load_upload_url( $upload_part['uploadUrl'] );
						}

						if ( isset( $upload_part['authorizationToken'] ) ) {
							$b2->load_authorization_token( $upload_part['authorizationToken'] );
						}
					}

					// Upload file chunk data
					$b2->upload_file( $file_chunk_data, $params['file_path'] );

					// Unset upload retries
					unset( $params['upload_retries'] );

				} catch ( Ai1wmae_Error_Exception $e ) {
					if ( $params['upload_retries'] <= 5 ) {
						return $params;
					}

					throw $e;
				}
			} else {
				if ( ! isset( $params['upload_url'] ) ) {
					try {

						$params['start_retries'] += 1;

						// Set file ID
						$file_id = $b2->start_large_file( $params['file_path'], $params['bucket_id'] );

						// Get new upload part URL
						if ( ( $upload_part = $b2->get_upload_part_url( $file_id ) ) ) {
							if ( isset( $upload_part['fileId'] ) ) {
								$params['file_id'] = $upload_part['fileId'];
							}

							if ( isset( $upload_part['uploadUrl'] ) ) {
								$params['upload_url'] = $upload_part['uploadUrl'];
							}

							if ( isset( $upload_part['authorizationToken'] ) ) {
								$params['upload_authorization_token'] = $upload_part['authorizationToken'];
							}
						}

						// Unset start retries
						unset( $params['start_retries'] );

					} catch ( Ai1wmae_Error_Exception $e ) {
						if ( $params['start_retries'] <= 5 ) {
							return $params;
						}

						throw $e;
					}
				}

				$b2->load_upload_url( $params['upload_url'] );
				$b2->load_authorization_token( $params['upload_authorization_token'] );

				try {

					$params['upload_retries'] += 1;
					$params['upload_backoff'] *= 2;

					// Upload file chunk data
					$file_chunk_sha1 = $b2->upload_part( $file_chunk_data, $params['file_chunk_number'] );

					// Add file chunk SHA1
					if ( ( $multipart = fopen( ai1wm_multipart_path( $params ), 'cb' ) ) ) {
						if ( fseek( $multipart, $params['multipart_offset'] ) !== -1 ) {
							fwrite( $multipart, $file_chunk_sha1 . PHP_EOL );
						}

						$params['multipart_offset'] = ftell( $multipart );

						fclose( $multipart );
					}

					// Set file chunk number
					$params['file_chunk_number'] += 1;

					// Unset upload retries
					unset( $params['upload_retries'] );
					unset( $params['upload_backoff'] );

				} catch ( Ai1wmae_Error_Exception $e ) {
					try {

						$b2->authorize_account();

						// Set upload backoff
						if ( isset( $params['upload_backoff'] ) ) {
							sleep( $params['upload_backoff'] );
						}

						// Get new upload part URL (503 Service Unavailable)
						if ( ( $upload_part = $b2->get_upload_part_url( $params['file_id'] ) ) ) {

							// Set file ID
							if ( isset( $upload_part['fileId'] ) ) {
								$params['file_id'] = $upload_part['fileId'];
							}

							// Set upload URL
							if ( isset( $upload_part['uploadUrl'] ) ) {
								$params['upload_url'] = $upload_part['uploadUrl'];
							}

							// Set upload authorization token
							if ( isset( $upload_part['authorizationToken'] ) ) {
								$params['upload_authorization_token'] = $upload_part['authorizationToken'];
							}
						}
					} catch ( Ai1wmae_Error_Exception $e ) {
					}

					if ( $params['upload_retries'] <= 5 ) {
						return $params;
					}

					throw $e;
				}
			}

			// Set archive offset
			$params['archive_offset'] = ftell( $archive );

			// Set archive details
			$name = ai1wm_archive_name( $params );
			$size = ai1wm_archive_size( $params );

			// Get progress
			$progress = (int) ( ( $params['archive_offset'] / $params['archive_size'] ) * 100 );

			// Set progress
			if ( defined( 'WP_CLI' ) ) {
				WP_CLI::log(
					sprintf(
						__( 'Uploading %s (%s) [%d%% complete]', AI1WMAE_PLUGIN_NAME ),
						$name,
						$size,
						$progress
					)
				);
			} else {
				Ai1wm_Status::info(
					sprintf(
						__(
							'<i class="ai1wmae-icon-b2"></i> ' .
							'Uploading <strong>%s</strong> (%s)<br />%d%% complete',
							AI1WMAE_PLUGIN_NAME
						),
						$name,
						$size,
						$progress
					)
				);
			}
		} else {

			// Add file chunk SHA1
			if ( isset( $params['file_id'] ) ) {
				$file_chunks = array();
				if ( ( $multipart = fopen( ai1wm_multipart_path( $params ), 'r' ) ) ) {
					while ( $file_chunk_sha1 = trim( fgets( $multipart ) ) ) {
						$file_chunks[] = $file_chunk_sha1;
					}

					fclose( $multipart );
				}

				// Finish upload file chunk data
				$b2->finish_large_file( $file_chunks, $params['file_id'] );

				// Unset file ID
				unset( $params['file_id'] );
			}

			// Set last backup date
			update_option( 'ai1wmae_b2_timestamp', time() );

			// Unset upload URL
			unset( $params['upload_url'] );

			// Unset upload authorization token
			unset( $params['upload_authorization_token'] );

			// Unset archive offset
			unset( $params['archive_offset'] );

			// Unset multipart offset
			unset( $params['multipart_offset'] );

			// Unset archive size
			unset( $params['archive_size'] );

			// Unset file chunk number
			unset( $params['file_chunk_number'] );

			// Unset completed
			unset( $params['completed'] );
		}

		// Close the archive file
		fclose( $archive );

		return $params;
	}
}
